import numpy as np
from numpy.linalg import norm
from scipy.optimize import minimize
import json
import sys

############################################################## Loading

def LoadPositions(filename):
	with open(filename, 'rt') as f:
		c=json.loads(f.read())

	positions = np.zeros((len(c), 3))
	indices = np.zeros((len(c)), dtype=np.int)
	for i in range(len(c)):
		indices[i] = int(c[i]['id'])
		positions[i,:] = c[i]['position']
	return positions, indices

# file names should contain the .json extension
def Load(referenceFilename, reconstructionFilename):
	ref, x = LoadPositions(referenceFilename)
	rec, i = LoadPositions(reconstructionFilename)
	completeness = len(i)/len(x)
	return ref[i], rec, completeness


############################################################## Comparing

def PointDistances(A, B):
	assert A.shape == B.shape
	return norm(A-B, axis=1)

def TrajectoryDistance(A, B):
	D = PointDistances(A,B)
	n = A.shape[0]
	return np.sqrt(1/n * np.sum(D**2))

# returns offset o such that A ~= B+o
def ComputeOffset(A, B):
	def f(p, A, B):
		return TrajectoryDistance(A-p, B)
	res = minimize(f, np.zeros((3)), args=(A, B))
	return res.x

	
	


doc = \
'''command line arguments: [ReferenceFile] [SubmissionFile]
Example: python TrackingPos.py DataBase/Tracking/GolemPathMove.json Users/ID404/GolemPathMove.json'''
if __name__ == "__main__":
	if len(sys.argv) != 3:
		print("invalid call:\n", doc)
		sys.exit(2) # the 2 sometimes signals command line syntax errors
	
	ref, rec, completeness = Load(sys.argv[1], sys.argv[2])
	
	offset = ComputeOffset(ref, rec)
	originalDistance = TrajectoryDistance(ref, rec)
	minimizedDistance= TrajectoryDistance(ref-offset, rec)
	
	output = {
	"status": "success",
	"completeness": completeness,
	"offset": norm(offset),
	"original_distance": originalDistance,
	"path_distance": minimizedDistance
	}
	
	print(json.dumps(output, indent="\t"))