function [data, header] = TransientImage(fullFilename)
% Reads a .ti file and returns the transient image data array as a 3D matrix
% and a header with information about the file and dataset. 
% Example usage:
% [transient_image, header_info] = TransientImage('filepath/here.ti');
% transient_image = TransientImage('filepath/here.ti');
 
	% Check syntax.  Must have at least one input argument, the full filename.
	if (nargin ~= 1)
		error('Requires a string with filename as argument.');
	end
	if (ischar(fullFilename) ~= 1)
		error('fullFilename is not a string');		
	end
	
	% Open the file for reading.
	file = fopen(fullFilename, 'rb', 'l', 'UTF-8');
	if (file == -1)
		error(['Error opening ', fullFilename, ' for input.']);
	end
	
	% Go to the beginning of the file.
	fseek(file, 0, 'bof');
	
	% Read MagicNumber (TI00 or TI01)
	header.magicNumber = [fread(file, 1, '*char'), fread(file, 1, '*char'), ...
		fread(file, 1, '*char'), fread(file, 1, '*char')];
	
	% select right version
	if(header.magicNumber == ['T', 'I', '0', '0']) | (header.magicNumber == ['T', 'I', '0', '1'])
		% old file version
		
		if (header.magicNumber == ['T', 'I', '0', '0'])
			header.uResolution = fread(file, 1, '*uint32');
			header.vResolution = fread(file, 1, '*uint32');
			header.numBins = fread(file, 1, '*uint32');
		elseif (header.magicNumber == ['T', 'I', '0', '1'])
			header.numBins = fread(file, 1, '*uint32');
			header.uResolution = fread(file, 1, '*uint32');
			header.vResolution = fread(file, 1, '*uint32');
		end
		header.numPixels = header.uResolution * header.vResolution;

		header.tMin = fread(file, 1, '*float32');
		header.tMax = fread(file, 1, '*float32');
		header.tDelta = (header.tMax-header.tMin) / single(header.numBins);

		data = fread(file, header.numPixels*header.numBins, '*float32');
		data = reshape(data, [header.numBins, header.uResolution, header.vResolution]);
		data = permute(data, [3, 2, 1]);
		
	elseif(header.magicNumber == ['T', 'I', '0', '4'])
		% new file version
		
		% file header
		header.pixelMode	= fread(file, 1, '*uint32');
		header.numPixels	= fread(file, 1, '*uint32');
		header.numBins		= fread(file, 1, '*uint32');
		header.tMin			= fread(file, 1, '*float32');
		header.tDelta		= fread(file, 1, '*float32');
		piBlockSize			= fread(file, 1, '*uint32');
		
		if(header.pixelMode ~= 10)
			error('pixelMode not supported')
		end
		
		% pixel data
		data = fread(file, header.numPixels*header.numBins, '*float32');
		
		% pixel interpretation
		header.uResolution = fread(file, 1, '*uint32')
		header.vResolution = fread(file, 1, '*uint32')
		header.topLeft = fread(file, 3, '*float32');
		header.topRight = fread(file, 3, '*float32');
		header.bottomLeft = fread(file, 3, '*float32');
		header.bottomRight = fread(file, 3, '*float32');
		header.laserPosition = fread(file, 3, '*float32');
		
		% rearange pixel data
		size(data)
		data = reshape(data, [header.numBins, header.uResolution, header.vResolution]);
		data = permute(data, [3, 2, 1]);
	else		
		error('Wrong header type');
	end
	
	fclose(file);
end